"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.initForElectronMain = exports.launchSafeBrowsingAndExit = void 0;
var electron_1 = require("electron");
var os_1 = __importDefault(require("os"));
var path_1 = __importDefault(require("path"));
var electron_channels_1 = require("./electron-channels");
var node_addon_shared_lib_loader_1 = require("@wontok/node-addon-shared-lib-loader");
var content_filtering_native_api_1 = require("./content-filtering-native-api");
var commands_1 = require("./commands");
var config_1 = __importDefault(require("./config"));
var utils_1 = require("./utils");
var currentState = null;
var launchSafeBrowsingAndExit = function () {
    utils_1.launchSafeBrowsing();
    electron_1.BrowserWindow.getAllWindows().forEach(function (window) { return window.close(); });
};
exports.launchSafeBrowsingAndExit = launchSafeBrowsingAndExit;
//#region Event Triggers
var notifyAllWindowsOfLatestState = function (publicOptions) {
    if (currentState != null) {
        publicOptions.onStateChange(currentState);
        electron_1.webContents.getAllWebContents().forEach(function (webContent) {
            webContent.send(electron_channels_1.Channels.contentFilteringOnStateChange, currentState);
        });
    }
};
var notifyAllWindowsOfBlockedContent = function (publicOptions, blockedContentInfo) {
    publicOptions.onBlockedContent(blockedContentInfo);
    electron_1.webContents.getAllWebContents().forEach(function (webContent) {
        webContent.send(electron_channels_1.Channels.contentFilteringOnBlockedContent, blockedContentInfo);
    });
};
//#endregion
var initForElectronMain = function (publicOptions) {
    var initOptions = {
        currentTimestamp: Math.floor(new Date().getTime() / 1000),
        contentFilteringServiceBinariesDirectory: config_1.default.contentFilteringServiceBinariesDirectory,
    };
    var onStateChangeCallback = function (event) {
        var secureDataStorePath;
        if (electron_1.app.isPackaged) {
            if (os_1.default.type() === "Windows_NT") {
                var programDataPath = process.env["ProgramData"];
                if (!programDataPath) {
                    throw new Error("programdata environment variable not defined!");
                }
                secureDataStorePath = path_1.default.resolve(programDataPath, electron_1.app.name, "SecureDataStore");
            }
            else if (os_1.default.type() === "Darwin") {
                secureDataStorePath = path_1.default.resolve("/Library/Application Support", electron_1.app.name, "SecureDataStore");
            }
            else {
                throw new Error("Unsupported OS: " + os_1.default.type());
            }
        }
        else {
            secureDataStorePath = path_1.default.resolve(process.cwd(), "run", "app-data");
        }
        var state = {
            secureDataStorePath: secureDataStorePath,
            blockedCategories: event.payload.blockedCategories,
            isFiltering: event.payload.isFiltering,
            licenceExpirationDate: new Date(event.payload.licenceExpirationTimestamp * 1000),
            nativeServiceAvailable: event.payload.nativeServiceAvailable,
        };
        currentState = state;
        utils_1.debug("Got new state in public Electron Land: " + JSON.stringify(state) + "\n");
        notifyAllWindowsOfLatestState(publicOptions);
    };
    onStateChangeCallback.callbackName = content_filtering_native_api_1.CallbackNames.onStateChange;
    var onBlockedContentCallback = function (event) {
        var blockedContentInfo = {
            categories: event.payload.categories,
            url: event.payload.url,
        };
        utils_1.debug("Got blocked content in public Electron Land: " + JSON.stringify(blockedContentInfo) + "\n");
        notifyAllWindowsOfBlockedContent(publicOptions, blockedContentInfo);
    };
    onBlockedContentCallback.callbackName = content_filtering_native_api_1.CallbackNames.onBlockedContent;
    var sharedLibController = node_addon_shared_lib_loader_1.initSharedLib(config_1.default.sharedLib, initOptions, [
        onStateChangeCallback,
        onBlockedContentCallback,
    ]);
    electron_1.ipcMain.on(electron_channels_1.Channels.contentFilteringOnRendererInit, function () {
        notifyAllWindowsOfLatestState(publicOptions);
    });
    electron_1.ipcMain.on(electron_channels_1.Channels.contentFilteringCommand, function (_, command) {
        sharedLibController.executeCommands([command]);
    });
    return {
        getOsName: utils_1.getOsName,
        launchSafeBrowsing: utils_1.launchSafeBrowsing,
        launchGhostpress: utils_1.launchGhostpress,
        terminateGhostpress: utils_1.terminateGhostpress,
        activateNativeService: function () {
            return sharedLibController.executeCommands([commands_1.commandActivateNativeService()]);
        },
        setBlockedCategories: function (categories) {
            return sharedLibController.executeCommands([
                commands_1.commandSetBlockedCategories(categories),
            ]);
        },
        setWhitelistedDomains: function (domains) {
            return sharedLibController.executeCommands([
                commands_1.commandSetWhitelistedDomains(domains),
            ]);
        },
        setLicenseExpirationDate: function (when) {
            return sharedLibController.executeCommands([
                commands_1.commandSetLicenseExpirationTimestamp(when),
            ]);
        },
        startFiltering: function () {
            return sharedLibController.executeCommands([commands_1.commandStartFiltering()]);
        },
        stopFiltering: function () {
            return sharedLibController.executeCommands([commands_1.commandStopFiltering()]);
        },
        updateTo: function (updateFilePath) {
            return sharedLibController.executeCommands([commands_1.commandUpdateTo(updateFilePath)]);
        },
        unload: sharedLibController.unload,
    };
};
exports.initForElectronMain = initForElectronMain;
